package ml.pluto7073.chemicals.handlers;

import com.mojang.brigadier.builder.LiteralArgumentBuilder;

import ml.pluto7073.chemicals.Chemicals;
import net.minecraft.class_1293;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2168;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2940;
import net.minecraft.class_2943;
import net.minecraft.class_2945;
import net.minecraft.class_2960;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;

/**
 * The base class representing a Chemical
 * <p>
 * Three pre-made handlers already exist:
 * <p>
 * {@link HalfLifeChemicalHandler}
 * <p>
 * The most realistic of the two.  Exponentially ticks down the amount of the chemical in the player's body
 * based on the half life in ticks specified
 * <p>
 * {@link LinearChemicalHandler}
 * <p>
 * The simplest handler.  Removed a set amount of chemical from the player each tick, until the amount reaches zero
 * <p>
 * {@link StaticChemicalHandler}
 * <p>
 * For chemicals that stay persistent in the player until death or a maximum amount is reached, where the latter
 * will perform an action on the player defined in {@link StaticChemicalHandler#onMaxAmountReached(class_1657)}
 */
public abstract class ConsumableChemicalHandler {

	public static final ConsumableChemicalHandler EMPTY =
			class_2378.method_10230(Chemicals.REGISTRY, Chemicals.id("empty"), new ConsumableChemicalHandler() {
				@Override
				public void tickPlayer(class_1657 player) {}

				@Override
				public Collection<class_1293> getEffectsForAmount(float amount, class_1937 level) {
					return List.of();
				}

				@Override
				public float get(class_1657 player) {
					return 0;
				}

				@Override
				public float add(class_1657 player, float amount) { return 0; }

				@Override
				public void set(class_1657 player, float amount) {}

				@Override
				public void defineDataForPlayer(class_2945 data) {}
			});

	protected final class_2940<Float> accessor;

	public ConsumableChemicalHandler() {
		accessor = class_2945.method_12791(class_1657.class, class_2943.field_13320);
	}

	/**
	 * Updates the current amount of the chemical in the specified player
	 * @param player The player to update
	 */
	public abstract void tickPlayer(class_1657 player);

	/**
	 * Gets the current amount of the chemical in the player
	 * @param player The player to retrieve
	 * @return Amount of the chemical
	 */
	public float get(class_1657 player) {
		return player.method_5841().method_12789(accessor);
	}

	/**
	 * Adds the chemical to a player
	 * @param player The player to add to
	 * @param amount The amount of the chemical to add
	 * @return The new amount of the chemical in the player
	 */
	public float add(class_1657 player, float amount) {
		float current = player.method_5841().method_12789(accessor);
		current += amount;
		if (current < 0) current = 0;
		player.method_5841().method_12778(accessor, current);
		return current;
	}

	/**
	 * Sets amount of chemical in a player
	 * @param player The player to set
	 * @param amount Amount of chemical
	 */
	public void set(class_1657 player, float amount) {
		player.method_5841().method_12778(accessor, Math.max(0f, amount));
	}

	/**
	 * Get a list of effects associated with an amount of this chemical
	 * @param amount Amount of the chemical
	 * @param level The current Level, for utility purposes
	 * @return A list of <code>MobEffectInstance</code>s
	 */
	public abstract Collection<class_1293> getEffectsForAmount(float amount, class_1937 level);

	public void defineDataForPlayer(class_2945 data) {
		data.method_12784(accessor, 0f);
	}

	/**
	 * Appends a tooltip to an item containing this chemical
	 * @param tooltip The current list of tooltips
	 * @param amount The amount in the stack
	 * @param stack The current stack, for utility purposes
	 */
	public void appendTooltip(List<class_2561> tooltip, float amount, class_1799 stack) {
		tooltip.add(class_2561.method_43469("tooltip.chemicals.amount", formatAmount(amount), class_2561.method_43471(getLanguageKey())));
	}

	public class_2960 getId() {
		return Chemicals.REGISTRY.method_29113(this).orElseThrow().method_29177();
	}

	public String getLanguageKey() {
		return getId().method_42093("chemical_handler");
	}

	/**
	 * Appends the units that this amount is tracked in to the amount, e.g. <code>amount + "mg"</code> or
	 * <code>amount + "L"</code>
	 */
	public String formatAmount(float amount) {
		return amount + "u";
	}

	/**
	 * Implement this method if you want custom syntax for this chemical's command using <code>/chemicals</code>
	 * @return A <code>LiteralArgumentBuilder</code> representing the entire subcommand for this chemical
	 * <p>
	 * <strong>Note: </strong> It is recommended that this begin with <code>literal("YOUR_CHEMICAL_ID")</code>
	 */
	public @Nullable LiteralArgumentBuilder<class_2168> createCustomChemicalCommandExtension() {
		return null;
	}

	/**
	 * Store any extra data required for the Chemical Handler
	 * @param player The player to store data from
	 * @param tag The CompoundTag to store the data in
	 */
	public void saveExtraPlayerData(class_1657 player, class_2487 tag) {}

	/**
	 * Load any extra data required for the Chemical Handler
	 * @param player The player to load to
	 * @param tag The tag to load from
	 */
	public void loadExtraPlayerData(class_1657 player, class_2487 tag) {}

}
