package ml.pluto7073.chemicals.handlers;

import com.mojang.brigadier.builder.LiteralArgumentBuilder;

import ml.pluto7073.chemicals.Chemicals;
import ml.pluto7073.chemicals.handlers.ConsumedInstance.AbsorptionType;
import ml.pluto7073.chemicals.item.ChemicalContaining;
import net.minecraft.class_1293;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2168;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2940;
import net.minecraft.class_2943;
import net.minecraft.class_2945;
import net.minecraft.class_2960;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector4f;

import java.util.Collection;
import java.util.List;

/**
 * The base class representing a Chemical
 * <p>
 * Three pre-made handlers already exist:
 * <p>
 * {@link HalfLifeChemicalHandler}
 * <p>
 * The most realistic of the two.  Exponentially ticks down the amount of the chemical in the player's body
 * based on the half life in ticks specified
 * <p>
 * {@link LinearChemicalHandler}
 * <p>
 * The simplest handler.  Removed a set amount of chemical from the player each tick, until the amount reaches zero
 * <p>
 * {@link StaticChemicalHandler}
 * <p>
 * For chemicals that stay persistent in the player until death or a maximum amount is reached, where the latter
 * will perform an action on the player defined in {@link StaticChemicalHandler#onMaxAmountReached(class_1657)}
 */
public abstract class ChemicalHandler {

	public static final ChemicalHandler EMPTY =
			class_2378.method_10230(Chemicals.CHEMICAL_HANDLER, Chemicals.id("air"), new ChemicalHandler(0) {
				@Override
				public void doTick(class_1657 player) {}

				@Override
				public Collection<class_1293> getEffectsForAmount(float amount, class_1937 level) {
					return List.of();
				}

				@Override
				public float add(class_1657 player, float amount) {
					return 0;
				}
			});

	protected final class_2940<Float> accessor;
	protected final class_2940<Integer> ticksAccessor;
	protected final float maxRecommendedAmount;

	public ChemicalHandler(float maxRecommendedAmount) {
		accessor = class_2945.method_12791(class_1657.class, class_2943.field_13320);
		ticksAccessor = class_2945.method_12791(class_1657.class, class_2943.field_13327);
		this.maxRecommendedAmount = maxRecommendedAmount;
	}

	public final void tickPlayer(class_1657 player) {
		int ticks = player.method_5841().method_12789(ticksAccessor) + 1;
		doTick(player);
		if (get(player) == 0) {
			player.method_5841().method_12778(ticksAccessor, 0);
			return;
		}
		player.method_5841().method_12778(ticksAccessor, ticks);
	}

	/**
	 * @return the length in ticks that <code>player</code> has had this chemical in their system.  Resets to 0
	 * if amount equals 0.
	 */
	public int getTicksInPlayer(class_1657 player) {
		return player.method_5841().method_12789(ticksAccessor);
	}

	/**
	 * Updates the current amount of the chemical in the specified player
	 * @param player The player to update
	 */
	protected abstract void doTick(class_1657 player);

	/**
	 * Gets the current amount of the chemical in the player
	 * @param player The player to retrieve
	 * @return Amount of the chemical
	 */
	public float get(class_1657 player) {
		return player.method_5841().method_12789(accessor);
	}

	/**
	 * @return The relative strength of the chemical in the specified <code>player</code> between 0 and 1, using the <code>maxRecommendedAmount</code>
	 */
	public float getStrength(class_1657 player) {
		return get(player) / maxRecommendedAmount;
	}

	/**
	 * Adds the chemical to a player
	 * @return The new amount of the chemical in the player
	 * @deprecated Use {@link class_1657#addChemical(ChemicalHandler, AbsorptionType, float)}
	 * or {@link class_1657#addChemical(ConsumedInstance)} to properly add chemicals to a player
	 */
	public float add(class_1657 player, float amount) {
		float current = player.method_5841().method_12789(accessor);
		current += amount;
		if (current < 0) current = 0;
		player.method_5841().method_12778(accessor, current);
		return current;
	}

	/**
	 * Sets amount of chemical in a player
	 */
	public void set(class_1657 player, float amount) {
		player.method_5841().method_12778(accessor, Math.max(0f, amount));
	}

	/**
	 * Get a list of effects associated with an amount of this chemical
	 * @param amount Amount of the chemical
	 * @param level The current Level, for utility purposes
	 * @return A list of <code>MobEffectInstance</code>s
	 */
	public abstract Collection<class_1293> getEffectsForAmount(float amount, class_1937 level);

	public void defineDataForPlayer(class_2945 data) {
		data.method_12784(accessor, 0f);
		data.method_12784(ticksAccessor, 0);
	}

	/**
	 * Appends a tooltip to an item containing this chemical
	 * @param tooltip The current list of tooltips
	 * @param amount The amount in the stack
	 * @param stack The current stack, for utility purposes
	 */
	public void appendTooltip(List<class_2561> tooltip, float amount, class_1799 stack) {
		tooltip.add(class_2561.method_43469("tooltip.chemicals.amount", formatAmount(amount), class_2561.method_43471(getLanguageKey())));
	}

	public class_2960 getId() {
		return Chemicals.CHEMICAL_HANDLER.method_29113(this).orElseThrow().method_29177();
	}

	public String getLanguageKey() {
		return getId().method_42093("chemical_handler");
	}

	/**
	 * Appends the units that this amount is tracked in to the amount, e.g. <code>amount + "mg"</code> or
	 * <code>amount + "L"</code>
	 */
	public String formatAmount(float amount) {
		return amount + "u";
	}

	/**
	 * @return the maximum amount of this drug in units that the player should logically have in their system.
	 * This will not scale with any settings/game rules and doesn't serve as a hard limit, just a way to assign
	 * a percentage to the amount in a player's system.
	 */
	public float getMaxRecommendedAmount() {
		return maxRecommendedAmount;
	}

	/**
	 * Implement this method if you want custom syntax for this chemical's command using <code>/chemicals</code>
	 * @return A <code>LiteralArgumentBuilder</code> representing the entire subcommand for this chemical
	 * <p>
	 * <strong>Note: </strong> It is recommended that this begin with <code>literal("example:your_chemical")</code>
	 */
	public @Nullable LiteralArgumentBuilder<class_2168> createCustomChemicalCommandExtension() {
		return null;
	}

	/**
	 * Store any extra data required for the Chemical Handler
	 * @param player The player to store data from
	 * @param tag The CompoundTag to store the data in
	 */
	public void saveExtraPlayerData(class_1657 player, class_2487 tag) {
		int ticks = player.method_5841().method_12789(ticksAccessor);
		if (ticks > 0) {
			tag.method_10569(getId().toString() + "/ticks", ticks);
		}
	}

	/**
	 * Load any extra data required for the Chemical Handler
	 * @param player The player to load to
	 * @param tag The tag to load from
	 */
	public void loadExtraPlayerData(class_1657 player, class_2487 tag) {
		if (tag.method_10545(getId().toString() + "/ticks")) {
			int ticks = tag.method_10550(getId().toString() + "/ticks");
			player.method_5841().method_12778(ticksAccessor, ticks);
		}
	}

	/**
	 * Creates a new <code>ConsumedInstance</code> using this chemical handler.
	 * @param type The desired speed of absorption
	 * @param amount The total amount of the chemical to add
	 * @return The new consumed instance
	 */
	public ConsumedInstance createInstance(AbsorptionType type, float amount) {
		return new ConsumedInstance(this, type, amount);
	}

	/**
	 * Creates a new <code>ConsumedInstance</code> using this chemical handler and
	 * the specified item as an instance of {@link ChemicalContaining}.
	 * @param type The desired speed of absorption
	 * @param stack The item stack to get information from
	 * @return The new consumed instance
	 */
	public ConsumedInstance createInstance(AbsorptionType type, class_1799 stack, class_1937 level) {
		if (!(stack.method_7909() instanceof ChemicalContaining item)) return new ConsumedInstance(this, type, 0);
		return new ConsumedInstance(this, type, item.getChemicalContent(getId(), stack, level));
	}

	public void contrast(Vector4f rgba, class_1657 player) {}

	public void bloom(Vector4f rgba, class_1657 player) {}

	public static void init() {}

}
