package ml.pluto7073.pdapi.addition;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import ml.pluto7073.pdapi.PDAPI;
import ml.pluto7073.pdapi.PDRegistries;
import ml.pluto7073.pdapi.addition.action.OnDrinkAction;
import ml.pluto7073.pdapi.addition.action.OnDrinkSerializer;
import ml.pluto7073.pdapi.addition.chemicals.ConsumableChemicalRegistry;
import ml.pluto7073.pdapi.networking.NetworkingUtils;
import net.minecraft.class_1309;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public class DrinkAddition {

    private final OnDrinkAction[] actions;
    private final boolean changesColor;
    private final int color;
    private final Map<String, Integer> chemicals;
    private final int maxAmount;
    private final int currentWeight;
    private final String name;

    protected DrinkAddition(OnDrinkAction[] actions, boolean changesColor, int color, Map<String, Integer> chemicals, int maxAmount, @Nullable String name) {
        this(actions, changesColor, color, chemicals, maxAmount, name, 0);
    }

    protected DrinkAddition(OnDrinkAction[] actions, boolean changesColor, int color, Map<String, Integer> chemicals, int maxAmount, @Nullable String name, int currentWeight) {
        this.actions = actions;
        this.changesColor = changesColor;
        this.color = color;
        this.chemicals = chemicals;
        this.maxAmount = maxAmount;
        this.currentWeight = currentWeight;
        this.name = name;
    }

    public void onDrink(class_1799 stack, class_1937 level, class_1309 user) {
        for (OnDrinkAction action : actions) {
            action.onDrink(stack, level, user);
        }
    }

    public boolean changesColor() {
        return changesColor;
    }

    public int getColor() {
        return color;
    }

    public Map<String, Integer> getChemicals() {
        return chemicals;
    }

    public int getMaxAmount() {
        return maxAmount;
    }

    public int getCurrentWeight() {
        return currentWeight;
    }

    public void toNetwork(class_2540 buf) {
        NetworkingUtils.writeDrinkActionsList(buf, actions);
        buf.writeBoolean(changesColor);
        buf.writeInt(color);
        buf.method_34063(chemicals, class_2540::method_10814, class_2540::writeInt);
        buf.writeInt(maxAmount);
        buf.writeInt(currentWeight);
        buf.method_10814(Objects.requireNonNullElse(name, ""));
    }

    public static DrinkAddition fromNetwork(class_2540 buf) {
        List<OnDrinkAction> actions = NetworkingUtils.readDrinkActionsList(buf);
        boolean changesColor = buf.readBoolean();
        int color = buf.readInt();
        Map<String, Integer> chemicals = buf.method_34067(class_2540::method_19772, class_2540::readInt);
        int maxAmount = buf.readInt();
        int currentWeight = buf.readInt();
        String name = buf.method_19772();
        if (name.isEmpty()) name = null;
        return new DrinkAddition(actions.toArray(OnDrinkAction[]::new), changesColor, color, chemicals, maxAmount, name, currentWeight);
    }

    public JsonObject toJson() {
        JsonObject json = new JsonObject();
        if (changesColor) {
            json.addProperty("changesColor", true);
            json.addProperty("color", color);
        }
        if (maxAmount > 0) {
            json.addProperty("maxAmount", maxAmount);
        }
        chemicals.forEach((id, amount) -> {
            if (amount > 0) {
                json.addProperty(id, amount);
            }
        });
        if (currentWeight != 0) {
            json.addProperty("weight", currentWeight);
        }
        if (name != null) {
            json.addProperty("name", name);
        }
        JsonArray actions = new JsonArray();
        for (OnDrinkAction action : this.actions) {
            JsonObject a = new JsonObject();
            @SuppressWarnings("unchecked")
            OnDrinkSerializer<OnDrinkAction> serializer = (OnDrinkSerializer<OnDrinkAction>) action.serializer();
            class_2960 id = PDRegistries.ON_DRINK_SERIALIZER.method_10221(serializer);
            if (id == null) {
                PDAPI.LOGGER.error("Couldn't serialize OnDrink Action");
                continue;
            }
            a.addProperty("type", id.toString());
            serializer.toJson(a, action);
            actions.add(a);
        }
        json.add("onDrinkActions", actions);
        return json;
    }

    public String getTranslationKey() {
        if (name != null) return name;
        try {
            class_2960 id = DrinkAdditions.getId(this);
            return id.method_42093("drink_addition");
        } catch (IllegalArgumentException e) {
            PDAPI.LOGGER.error("Couldn't get translation key for a drink addition", e);
            return "drink_addition.pdapi.empty";
        }
    }

    @SuppressWarnings("UnusedReturnValue")
    public static class Builder {

        private final List<OnDrinkAction> actions;
        private boolean changesColor;
        private int color;
        private final HashMap<String, Integer> chemicals;
        private int maxAmount;
        private int weight;
        private String name;

        public Builder() {
            actions = new ArrayList<>();
            changesColor = false;
            color = 0;
            chemicals = new HashMap<>();
            ConsumableChemicalRegistry.forEach(handler -> chemical(handler.getName(), 0));
            maxAmount = 0;
            weight = 0;
            name = null;
        }

        public Builder addAction(OnDrinkAction action) {
            actions.add(action);
            return this;
        }

        public Builder changesColor(boolean changesColor) {
            this.changesColor = changesColor;
            return this;
        }

        public Builder color(int color) {
            this.color = color;
            return this;
        }

        public Builder chemical(String name, int amount) {
            chemicals.put(name, amount);
            return this;
        }

        public Builder maxAmount(int amount) {
            this.maxAmount = amount;
            return this;
        }

        public Builder setWeight(int weight) {
            this.weight = weight;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public DrinkAddition build() {
            return new DrinkAddition(actions.toArray(OnDrinkAction[]::new), changesColor, color, chemicals, maxAmount, name, weight);
        }

    }

}
