package ml.pluto7073.pdapi.networking;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import ml.pluto7073.pdapi.PDAPI;
import ml.pluto7073.pdapi.PDRegistries;
import ml.pluto7073.pdapi.addition.DrinkAddition;
import ml.pluto7073.pdapi.addition.action.OnDrinkAction;
import ml.pluto7073.pdapi.addition.action.OnDrinkSerializer;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import java.util.*;
import java.util.function.Function;

public final class NetworkingUtils {

    private static final int INT_TYPE = 0,
            FLOAT_TYPE = 1,
            BOOLEAN_TYPE = 2,
            STRING_TYPE = 3,
            ARRAY_BEGIN_TYPE = 4,
            ARRAY_END_TYPE = 5,
            OBJECT_BEGIN_TYPE = 6,
            OBJECT_END_TYPE = 7,
            KEY_TYPE = 8;

    public static JsonObject readJsonObject(class_2540 buffer) {
        Stack<Integer> stack = new Stack<>();
        stack.push(OBJECT_BEGIN_TYPE);
        int i = buffer.readInt();
        if (i != OBJECT_BEGIN_TYPE) throw new IllegalStateException("Expected object initializer as first signal");
        return readJsonObject(buffer, stack);
    }

    private static JsonObject readJsonObject(class_2540 buffer, Stack<Integer> objectLevels) {
        JsonObject object = new JsonObject();
        String currentKey = "";
        loop: while (true) {

            int signal = buffer.readInt();

            switch (signal) {
                case INT_TYPE -> object.add(currentKey, new JsonPrimitive(buffer.readInt()));
                case FLOAT_TYPE -> object.add(currentKey, new JsonPrimitive(buffer.readFloat()));
                case BOOLEAN_TYPE -> object.add(currentKey, new JsonPrimitive(buffer.readBoolean()));
                case STRING_TYPE -> object.add(currentKey, new JsonPrimitive(buffer.method_19772()));
                case ARRAY_BEGIN_TYPE -> {
                    objectLevels.push(ARRAY_BEGIN_TYPE);
                    object.add(currentKey, readJsonArray(buffer, objectLevels));
                    objectLevels.pop();
                }
                case ARRAY_END_TYPE -> throw new IllegalStateException("Unexpected array ending");
                case OBJECT_BEGIN_TYPE -> {
                    objectLevels.push(OBJECT_BEGIN_TYPE);
                    object.add(currentKey, readJsonObject(buffer, objectLevels));
                    objectLevels.pop();
                }
                case OBJECT_END_TYPE -> {
                    if (objectLevels.isEmpty()) throw new IllegalStateException("Unexpected object ending");
                    if (objectLevels.peek() != OBJECT_BEGIN_TYPE) throw new IllegalStateException("Unexpected object ending");
                    break loop;
                }
                case KEY_TYPE -> currentKey = buffer.method_19772();
            }
        }
        return object;
    }

    private static JsonArray readJsonArray(class_2540 buffer, Stack<Integer> objectLevels) {
        JsonArray array = new JsonArray();
        loop: while (true) {

            int signal = buffer.readInt();

            switch (signal) {
                case INT_TYPE -> array.add(buffer.readInt());
                case FLOAT_TYPE -> array.add(buffer.readFloat());
                case BOOLEAN_TYPE -> array.add(buffer.readBoolean());
                case STRING_TYPE -> array.add(buffer.method_19772());
                case ARRAY_BEGIN_TYPE -> {
                    objectLevels.push(ARRAY_BEGIN_TYPE);
                    array.add(readJsonArray(buffer, objectLevels));
                    objectLevels.pop();
                }
                case ARRAY_END_TYPE -> {
                    if (objectLevels.isEmpty()) throw new IllegalStateException("Unexpected Array Ending");
                    if (objectLevels.peek() != ARRAY_BEGIN_TYPE) throw new IllegalStateException("Unexpected Array Ending");
                    break loop;
                }
                case OBJECT_BEGIN_TYPE -> {
                    objectLevels.push(OBJECT_BEGIN_TYPE);
                    array.add(readJsonObject(buffer, objectLevels));
                    objectLevels.pop();
                }
                case OBJECT_END_TYPE -> throw new IllegalStateException("Unexpected Object Ending");
                case KEY_TYPE -> throw new IllegalStateException("Unexpected Key Type");
            }
        }
        return array;
    }

    public static void writeJsonObjectStart(class_2540 buf, JsonObject object) {
        buf.method_53002(OBJECT_BEGIN_TYPE);
        writeJsonObject(buf, object);
    }

    private static void writeJsonObject(class_2540 buf, JsonObject object) {
        for (Map.Entry<String, JsonElement> entry : object.entrySet()) {
            buf.method_53002(KEY_TYPE);
            buf.method_10814(entry.getKey());

            JsonElement value = entry.getValue();
            writeValue(value, buf);
        }

        buf.method_53002(OBJECT_END_TYPE);
    }

    private static void writeJsonArray(class_2540 buf, JsonArray array) {
        for (JsonElement value : array) {
            writeValue(value, buf);
        }
        buf.method_53002(ARRAY_END_TYPE);
    }

    private static void writeValue(JsonElement value, class_2540 buf) {
        if (value.isJsonArray()) {
            buf.method_53002(ARRAY_BEGIN_TYPE);
            writeJsonArray(buf, value.getAsJsonArray());
        } else if (value.isJsonObject()) {
            buf.method_53002(OBJECT_BEGIN_TYPE);
            writeJsonObject(buf, value.getAsJsonObject());
        } else if (value.isJsonPrimitive()) {
            JsonPrimitive primVal = value.getAsJsonPrimitive();
            if (primVal.isBoolean()) {
                buf.method_53002(BOOLEAN_TYPE);
                buf.method_52964(primVal.getAsBoolean());
            } else if (primVal.isNumber()) {
                if (primVal.getAsNumber().intValue() == primVal.getAsNumber().floatValue()) {
                    buf.method_53002(INT_TYPE);
                    buf.method_53002(primVal.getAsInt());
                } else {
                    buf.method_53002(FLOAT_TYPE);
                    buf.method_52941(primVal.getAsFloat());
                }
            } else if (primVal.isString()) {
                buf.method_53002(STRING_TYPE);
                buf.method_10814(primVal.getAsString());
            }
        }
    }

    public static <T> JsonObject[] convertToJson(T[] array, Function<T, JsonObject> converter) {
        JsonObject[] result = new JsonObject[array.length];
        for (int i = 0; i < result.length; i++) {
            result[i] = converter.apply(array[i]);
        }
        return result;
    }

    public static <T> void arrayToNetwork(class_2540 buf, T[] array, class_2540.class_7462<T> itemWriter) {
        HashMap<Integer, T> intMap = new HashMap<>();
        for (int i = 0; i < array.length; i++) {
            intMap.put(i, array[i]);
        }
        buf.method_34063(intMap, class_2540::method_53002, itemWriter);
    }

    public static <T> List<T> listFromNetwork(class_2540 buf, class_2540.class_7461<T> itemReader) {
        Map<Integer, T> intMap = buf.method_34067(class_2540::readInt, itemReader);
        List<T> list = new ArrayList<>();
        for (int i = 0; i < intMap.size(); i++) {
            list.add(intMap.get(i));
        }
        return list;
    }

    public static List<OnDrinkAction> readDrinkActionsList(class_2540 buf) {
        return listFromNetwork(buf, b -> {
            class_2960 id = b.method_10810();
            @SuppressWarnings("unchecked")
            OnDrinkSerializer<OnDrinkAction> serializer = (OnDrinkSerializer<OnDrinkAction>)
                    PDRegistries.ON_DRINK_SERIALIZER.method_10223(id);
            if (serializer == null) throw new IllegalStateException();
            return serializer.fromNetwork(b);
        });
    }

    public static void writeDrinkActionsList(class_2540 buf, OnDrinkAction[] actions) {
        arrayToNetwork(buf, actions, (b, action) -> {
            @SuppressWarnings("unchecked")
            OnDrinkSerializer<OnDrinkAction> serializer = (OnDrinkSerializer<OnDrinkAction>) action.serializer();
            class_2960 id = PDRegistries.ON_DRINK_SERIALIZER.method_10221(serializer);
            b.method_10812(id);
            serializer.toNetwork(b, action);
        });
    }

}
